<?php
/**
 * Methods for the Simple Links Plugin
 *
 * @author OnPoint Plugins <support@onpointplugins.com>
 *
 *
 * @uses   These methods are used in both the admin output of the site
 *
 * @see    simple_links_admin() for the only admin methods
 */

if ( ! class_exists( 'simple_links' ) ) {
	class simple_links {

		public $additional_fields = array();


		public function __construct() {
			$this->register_shortcode();

			//Add the translate ability
			add_action( 'plugins_loaded', array( $this, 'load_translations' ) );

			//Setup the form output for the new button
			add_filter( 'query_vars', array( $this, 'add_shortcode_form_query_var' ) );
			add_action( 'template_redirect', array( $this, 'load_shortcode_form' ), 1 );

			//Add the widgets
			add_action( 'widgets_init', array( $this, 'add_widgets' ) );

		}


		/**
		 * Some magic to bring back deprecated methods which do not
		 * pass phpcs standards and therefore may not be included.
		 *
		 * @param $name
		 * @param $arguments
		 *
		 * @todo Formally deprecate anything here
		 *
		 * @throws BadMethodCallException
		 *
		 * @return array
		 */
		public function __call( $name, $arguments ) {
			//used by CSV Import/Export
			if ( 'getAdditionalFields' === $name ) {
				return $this->get_additional_fields();
			}

			throw new BadMethodCallException( __( 'Non Existent method in simple_links class', 'simple-links' ) );
		}



		public function register_shortcode() {
			add_shortcode( 'simple-links', array( $this, 'shortcode' ) );
		}


		/**
		 * orderby_options
		 *
		 * @param string $selected
		 *
		 * @static
		 * @return void
		 */
		public static function orderby_options( $selected = null ) {
			?>
			<option value="menu_order" <?php selected( $selected, 'menu_order' ); ?>>
				<?php esc_html_e( 'Link Order', 'simple-links' ); ?>
			</option>
			<option value="title" <?php selected( $selected, 'title' ); ?>>
				<?php esc_html_e( 'Title', 'simple-links' ); ?>
			</option>
			<option value="rand" <?php selected( $selected, 'rand' ); ?>>
				<?php esc_html_e( 'Random', 'simple-links' ); ?>
			</option>
			<option value="date" <?php selected( $selected, 'date' ); ?>>
				<?php esc_html_e( 'Date', 'simple-links' ); ?>
			</option>
			<?php

		}


		/**
		 * Register the widgets
		 *
		 * @since 4.4.0
		 *
		 * @uses  added to the widgets_init hook by self::__construct();
		 */
		public function add_widgets() {
			//Register the main widget
			register_widget( 'Simple_Links__Widgets__Simple_Links' );

		}

		/**
		 * Add the translate ability for I18n standards
		 *
		 * @since 4.4.0
		 * @uses  called on __construct()
		 */
		public function load_translations() {
			load_plugin_textdomain( 'simple-links', false, 'simple-links/languages' );
		}


		/**
		 * Creates the shortcode output
		 *
		 * @param array  $attributes = 'title'              => false,
		 *                           'category'           => false,
		 *                     'orderby'           => 'menu_order',
		 *                     'count'             => '-1',
		 *                     'show_image'        => false,
		 *                     'show_image_only'   => false,
		 *                     'image_size'        => 'thumbnail',
		 *                     'order'             => 'ASC',
		 *                     'fields'            => false,
		 *                     'description'       => false,
		 *                     'separator'         =>  '-',
		 *                     'id'                =>  false,
		 *                     'remove_line_break' =>  false
		 *
		 * @filters
		 *       the shortcode atts
		 *      * add_filter( 'simple_links_shortcode_atts', $atts );
		 *       the shortcode output
		 *      * add_filter( 'simple_links_shortcode_output', $output, $links, $atts )
		 *       the links object directly
		 *      *  apply_filters('simple_links_shortcode_links_object', $links, $atts);
		 *       the links meta data per link
		 *      * apply_filters('simple_links_shortcode_link_meta', $meta, $link, $atts );
		 *
		 *
		 * @uses  [simple-links $atts]
		 *
		 * @uses  the function filtering this output can accept 3 args.   <br>
		 *                $output = The Output Generated by the Function
		 *                $links  = The complete links to direct munipulation
		 *                $atts   = The shortcode Attributes sent to this
		 * @uses  All filters may be used by id by calling them with the id appened like so  'simple_links_shortcode_output_%id%' there must be an 'id' specified in the shortcode for this to work
		 * @uses  Using the filters without the id will filter all the shortcodes
		 *
		 * @since 1.7.14
		 *
		 * @return string - the created list based on attributes
		 *
		 */
		public function shortcode( $attributes ) {
			$attributes = apply_filters( 'simple_links_shortcode_atts', $attributes );
			if ( isset( $attributes['id'] ) ) {
				$attributes = apply_filters( 'simple_links_shortcode_atts_' . $attributes['id'], $attributes );
			}

			$links = new SimpleLinksFactory( $attributes, 'shortcode' );

			$filter_params = array(
				$links->output(),
				$links->links,
				$links->args,
				$links->query_args,
			);
			$output        = apply_filters_ref_array( 'simple_links_shortcode_output', $filter_params );

			if ( isset( $attributes['id'] ) ) {
				$filter_params[0] = $output;
				$output           = apply_filters_ref_array( 'simple_links_shortcode_output_' . $attributes['id'], $filter_params );
			}

			return $output;

		}


		/**
		 * @deprecated use Simple_Links_Categories::get_category_names()
		 *
		 * @todo       find all uses of this and convert to new object
		 *
		 */
		public function get_categories() {
			return Simple_Links_Categories::get_category_names();

		}


		/**
		 * Retrieves all available image sizes
		 *
		 * @todo figure out a way to make this work on WP Vip
		 * @link https://vip.wordpress.com/documentation/vip-development-tips-tricks/image-resizing-and-cropping/#custom-image-sizes-and-custom-cropping
		 * @sniff WordPress.VIP.RestrictedFunctions.get_intermediate_image_sizes_get_intermediate_image_sizes
		 *
		 */
		public function image_sizes() {
			return get_intermediate_image_sizes();
		}


		/**
		 * Load Shortcode Form
		 *
		 * Loads the output of the shortcode form into the modal
		 *
		 * @uses added to the template_redirect hook by self::__construct();
		 *
		 * @uses called by the mce icon
		 *
		 * @return void
		 */
		public function load_shortcode_form() {
			$var = get_query_var( 'simple_links_shortcode' );
			if ( 'form' !== $var ) {
				return;
			}
			require_once ABSPATH . '/wp-admin/includes/template.php';

			wp_enqueue_style(
				'simple-links-shortcode-css',
				SIMPLE_LINKS_CSS_DIR . 'simple-links-shortcode.css',
				array(),
				SIMPLE_LINKS_VERSION
			);

			wp_enqueue_script(
				'mce_popup',
				includes_url() . 'js/tinymce/tiny_mce_popup.js',
				array( 'jquery' ),
				SIMPLE_LINKS_VERSION
			);

			wp_enqueue_script(
				'sl-shortcode-form',
				SIMPLE_LINKS_JS_DIR . 'shortcode-form.js',
				array( 'jquery', 'mce_popup' ),
				SIMPLE_LINKS_VERSION
			);

			wp_localize_script( 'sl-shortcode-form', 'Simple_Links_Config', simple_links_admin()->js_config() );

			do_action( 'simple-links/shortcode-form/before' );

			include SIMPLE_LINKS_DIR . '/admin-views/shortcode-form.php';

			do_action( 'simple-links/shortcode-form/after' );

			die();

		}


		/**
		 * Setsup the query var to bring in the outside page to the popup form
		 *
		 * @since 4.4.0
		 *
		 * @param array $query_vars
		 *
		 * @return array
		 */
		public function add_shortcode_form_query_var( $query_vars ) {
			$query_vars[] = 'simple_links_shortcode';
			return $query_vars;
		}


		/**
		 * Retrieve the additional fields names
		 *
		 * @since 4.4.0
		 */
		public function get_additional_fields() {
			return (array) get_option( 'link_additional_fields' );
		}

		/**
		 * Get the additional Field Values for a post
		 *
		 * @since 4.4.0
		 *
		 * @param int $post_id
		 *
		 * @return array
		 */
		public function get_additional_field_values( $post_id ) {
			return get_post_meta( $post_id, 'link_additional_value', true );
		}


		/**
		 * Retrieves all the link categories a link is assigned to
		 *
		 * @param int     $post_id            the link ID
		 * @param boolean $return_full_object to return all values default to an array of just names
		 *
		 * @return boolean|array
		 */
		public function get_link_categories( $post_id, $return_full_object = false ) {
			$cats = get_the_terms( $post_id, 'simple_link_category' );

			if ( empty( $cats ) ) {
				return false;
			}

			if ( $return_full_object ) {
				return $cats;
			}

			$cat_names = array();
			foreach ( $cats as $cat ) {
				$cat_names[] = $cat->name;
			}

			return $cat_names;
		}


	}
}
